package ga.examples.quadratic;

import ga.core.algorithm.util.ClusterUtil;
import ga.core.algorithm.util.RandomSingleton;
import ga.core.individual.IClusterableIndividual;
import ga.core.individual.IIntervalFitness;
import ga.core.validation.GAContext;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicLong;

public final class QuadraticClusterIndividual implements
    IClusterableIndividual<QuadraticClusterIndividual>, IIntervalFitness {
  public static final int MAX_NUM = 8;
  public static final int PARAMETER_COUNT = 3;

  private static final AtomicLong ID_GENERATOR = new AtomicLong();

  private final long id;

  private final List<Integer> genotype = new ArrayList<Integer>();
  private transient int result = Integer.MIN_VALUE;

  private final Random rnd = RandomSingleton.getRandom();

  private double min;
  private double max;
  private double center = UNEVALUATED;

  public QuadraticClusterIndividual() {
    id = ID_GENERATOR.incrementAndGet();
  }

  @Override
  public long getId() {
    return id;
  }

  @Override
  public void setContext(final GAContext context) {
  }

  @Override
  public GAContext getContext() {
    return null;
  }

  @Override
  public void initRandomly() {
    center = UNEVALUATED;
    genotype.clear();

    for (int i = 0; i < PARAMETER_COUNT; i++) {
      genotype.add(rnd.nextInt(MAX_NUM + 1));
    }
  }

  @Override
  public void setFitness(final double fitness) {
    this.center = fitness;
  }

  @Override
  public double getFitness() {
    if (!isEvaluated()) {
      throw new RuntimeException("Not evaluated individual");
    }
    return center;
  }

  public List<Integer> getGenotype() {
    return genotype;
  }

  @Override
  public String toString() {
    String s = genotype.get(0) + "^2*" + genotype.get(1) + "+"
        + genotype.get(2);

    if (isEvaluated()) {
      s += "=" + result + " Fitness: ";
      if (min != max) {
        s += "(" + min + "," + max + ")";
      } else {
        s += center;
      }
    } else {
      s += " unevaluated";
    }

    return s;
  }

  public void setResult(final int result) {
    this.result = result;
  }

  public int getResult() {
    return result;
  }

  @Override
  public boolean isEvaluated() {
    return center != UNEVALUATED;
  }

  @Override
  public QuadraticClusterIndividual clone() {
    final QuadraticClusterIndividual ind = new QuadraticClusterIndividual();
    ind.genotype.addAll(this.genotype);
    return ind;
  }

  @Override
  public double getFitnessWidth() {
    return max - min;
  }

  @Override
  public void setFitnessInterval(final double center, final double width) {
    this.center = center;
    min = center - (width / 2d);
    max = min + width;
  }

  @Override
  public void setFitnessLimits(final double min, final double max) {
    this.min = min;
    this.max = max;
    this.center = (max + min) / 2d;
  }

  @Override
  public double getMinFitness() {
    return min;
  }

  @Override
  public double getMaxFitness() {
    return max;
  }

  @Override
  public QuadraticClusterIndividual centroidOf(
      final Collection<QuadraticClusterIndividual> c) {
    return ClusterUtil.calculateCentroid(c);
  }

  @Override
  public double distanceFrom(final QuadraticClusterIndividual p) {
    return Math.abs(result - p.result);
  }

  public static void main(final String[] args) {
    final QuadraticClusterEvaluator ev = new QuadraticClusterEvaluator(42);

    final QuadraticClusterIndividual ind1 = new QuadraticClusterIndividual();
    ind1.getGenotype().add(1);
    ind1.getGenotype().add(2);
    ind1.getGenotype().add(3);
    ev.evaluate(ind1);

    System.err.println(ind1);

    final QuadraticClusterIndividual ind2 = new QuadraticClusterIndividual();
    ind2.getGenotype().add(3);
    ind2.getGenotype().add(2);
    ind2.getGenotype().add(3);
    ev.evaluate(ind2);

    System.err.println(ind2);

    final QuadraticClusterIndividual ind3 = new QuadraticClusterIndividual();
    ind3.getGenotype().add(1);
    ind3.getGenotype().add(3);
    ind3.getGenotype().add(3);
    ev.evaluate(ind3);

    System.err.println(ind3);

    System.err.println(ind1.distanceFrom(ind2));
    System.err.println(ind1.distanceFrom(ind3));
  }

}
